/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: Marshaller.cpp 7178 2015-07-24 12:36:25Z Robert $
   Last Modification:
    $Author: Robert $
    $Date: 2015-07-24 14:36:25 +0200 (Fr, 24 Jul 2015) $
    $Revision: 7178 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no

   Description:
     Implementation of the cifX API Marshaller class

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     10        24.07.15    RM       Adapted to new common header files
     9         16.04.15    LC       Bugfix: 64bit in CMarshaller::OpenChannel sizeof(size_t) delivers 8 Bytes
                                    instead of the expected 4 Bytes in header structure

     8         23.02.15    RM       Bugfix: xChannelIOReadSendData() not delivering I/O data
                                    because ulDataLen not set to user parameter and
                                    GeneratedHeader() passed a worng request length

     7         28.03.13    RM       Fixed xChannelIORead() and xChannelReadSendData() not delivering I/O data
                                    on COM_FLAG error return

     7         07.10.11    SS       Reduce the timeout of RPC involved in connection establishment

     6         18.04.11    SS       Remove parameter checks from cifX API calls, as this
                                    is already done by cifX API calling layer

     5         01.04.11    SS       Connection state added (m_fConnectionValid)

     4         29.06.10    SS       Bugfix: Hardcoded timeout of remote procedure calls
                                    may causes packet loss.

     3         25.05.10    SS       Validate status of packet reply to reject
                                    invalid packets from device side

     2         03.03.09    MT       Structural change to allow placing physical drivers
                                    in a dll

     1         25.09.08    PL       initial version

**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file Marshaller.cpp
/// Implementation of the cifX API Marshaller class
/////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "Marshaller.h"
#include "DeviceHandler.h"
#include "TransportLayer.h"
#include "PhysicalInterface.h"
#include "HelperFunctions.h"

/////////////////////////////////////////////////////////////////////////////
/// Constructor
///   \param pcTransportLayer  Transport layer which is use for sending data
///   \param pcEndpoint        Endpoint which is use for sending data
/////////////////////////////////////////////////////////////////////////////
CMarshaller::CMarshaller(CTransportLayer* pcTransportLayer,
                         CEndpoint*       pcEndpoint)
: CDataLayer(pcTransportLayer)
, m_pcEndpoint(pcEndpoint)
, m_dwClassFactory(0)
, m_ulVersion(0)
, m_dwDriverHandle(0)
, m_usSequenceNr(0)
, m_fSequenceNrSupport(true)
, m_hReConnectEvent(NULL)
{
  InitializeCriticalSection(&m_tDeviceMapLock);
  InitializeCriticalSection(&m_tcsResetLock);
/*
  if(NULL != pcTransport)
  {
    // Set Event or callback
    m_hReConnectEvent    = ::CreateEvent(NULL, FALSE, FALSE, NULL);
    if(NULL != m_hReConnectEvent)
    {
      bool fRet = pcTransport->RegisterReconnectEvent( this, HIL_TRANSPORT_TYPE_MARSHALLER, m_hReConnectEvent);
      if(false == fRet)
      {
        CloseHandle(m_hReConnectEvent);
        m_hReConnectEvent = NULL;
      }
    }
  }
  m_cmDevice.clear();
  */
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CMarshaller::~CMarshaller(void)
{
  // Check Reconnect event was set 
  /*if(NULL != m_hReConnectEvent)
  {
    m_pcTransport->UnRegisterReconnectEvent(this);  
    CloseHandle(m_hReConnectEvent);
    m_hReConnectEvent = NULL;
  } */


  Deinit();

  DeleteCriticalSection(&m_tcsResetLock);
  DeleteCriticalSection(&m_tDeviceMapLock);

}

/////////////////////////////////////////////////////////////////////////////
/// Data layer callback for incoming data (should never
/// be called, as we are only using transfers on transport layer)
/// Only reason it may be called is a timeout of a marshaller packet
///   \param  ptPacket Incoming data packet
///   \return true if packet should be deleted by caller
/////////////////////////////////////////////////////////////////////////////
bool CMarshaller::ReceiveData( PHIL_TRANSPORT_PACKET ptPacket)
{
  UNREFERENCED_PARAMETER( ptPacket);
  
  if(NULL == ptPacket)
    return false;
  
return true;
}

/////////////////////////////////////////////////////////////////////////////
/// Data layer callback when send has been acked (should never
/// be called, as we are only using transfers on transport layer)
///   \param  ptHeader   Header of packet that has been acknowledged
///   \return void
/////////////////////////////////////////////////////////////////////////////
void CMarshaller::SendComplete( PHIL_TRANSPORT_HEADER ptHeader)
{
  UNREFERENCED_PARAMETER( ptHeader);
  // We never should be here on run time , Marshaller don't use async data transfer
//  _ASSERT(false);
}

/////////////////////////////////////////////////////////////////////////////
/// Initialize Marshaller layer
///   \return CIFX_NO_ERROR  on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::Init(void)
{
  int32_t          lRet  = CIFX_DRV_INIT_ERROR;

  if( (CIFX_NO_ERROR == (lRet = QueryMarshallerVersion()))                           &&
      (CIFX_NO_ERROR == (lRet = CreateObject( 0, 
                                              MARSHALLER_OBJECT_TYPE_CLASSFACTORY, 
                                              m_dwClassFactory)))                    &&
      (CIFX_NO_ERROR == (lRet = CreateObject( m_dwClassFactory, 
                                              MARSHALLER_OBJECT_TYPE_DRIVER, 
                                              m_dwDriverHandle)))                    )
  {
    if(CIFX_NO_ERROR == (lRet = OpenDriver()))
    {
      uint32_t     ulBoard    = 0;
      BOARD_INFORMATION tBoardInfo = {0};

      /* Device supports packets, so insert it into available data layers */
      m_pcEndpoint->m_cmDataLayers[HIL_TRANSPORT_TYPE_MARSHALLER] = this;
      m_pcEndpoint->m_pcDefaultDataLayer                          = this;

      /* Enumerate channels, and create channel objects */
      while(CIFX_NO_ERROR == EnumDevice(ulBoard, sizeof(tBoardInfo), &tBoardInfo))
      {
        // Get the necessary names and store the information in the device
        CDevice* pcDevice = m_pcEndpoint->AddDevice(tBoardInfo);
        pcDevice->AddChannel(RCX_SYSTEM_CHANNEL, NULL);

        uint32_t       ulChannel    = 0;
        CHANNEL_INFORMATION tChannelInfo = {0};

        while(CIFX_NO_ERROR == EnumChannel(ulBoard, ulChannel, sizeof(tChannelInfo), &tChannelInfo))
        {
          pcDevice->AddChannel(ulChannel, &tChannelInfo);
          ++ulChannel;
        }

        ++ulBoard;
      }

      m_fConnectionValid = true;

      TRACE("%s: Datalayer initialized - cifX API Marshaller (V%d.%d.%d.%d)",
            m_pcEndpoint->m_pcInterface->GetInterfaceName().c_str(),
            m_ulVersion >> 24, 
            (m_ulVersion >> 16 )& 0xFF, 
            (m_ulVersion >> 8)  & 0xFF, 
            m_ulVersion & 0xFF);

    }
  }

  if(CIFX_NO_ERROR != lRet)
  {
    Deinit();

    TRACE("%s: Datalayer not supported - cifX API Marshaller",
          m_pcEndpoint->m_pcInterface->GetInterfaceName().c_str());
  }

  return lRet;
}
/////////////////////////////////////////////////////////////////////////////
/// Uninitialize data layer (drop all connections)
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
///  Uninitialize data layer, so destroy all list 
///  devices and channel for this layer
///   \return void
/////////////////////////////////////////////////////////////////////////////
void CMarshaller::Deinit(void)
{
  EnterCriticalSection(&m_tcsResetLock);
  
  while(m_cmDevice.size() > 0)
  {
    MARSH_DEVICE_MAP::iterator iterDev = m_cmDevice.begin();

    while(!iterDev->second.empty())
    {
      MARSH_CHANNEL_MAP& tChannel = iterDev->second;

      // Close all channels of a device 
      while(!tChannel.empty())
      {
        PMARSH_CHANNEL_INFO_T ptChannelInfo = tChannel.begin()->second;
        CloseChannel(ptChannelInfo, true, true);
      }
    }
    
    m_cmDevice.erase(iterDev);
  }

  if(m_dwDriverHandle != 0)
  {
    CloseDriver();
  }

  m_dwDriverHandle = 0;
  m_dwClassFactory = 0;
  m_fConnectionValid = false;

  LeaveCriticalSection(&m_tcsResetLock);

}

/////////////////////////////////////////////////////////////////////////////
/// Create an object on the remote end (e.g. 
/// Class factory / Driver object
///   \param dwDestHandle   Remote handle to send request to
///   \param dwObjectType   Type of object to create
///   \param dwObjectHandle Returned handle of the created object
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::CreateObject(DWORD   dwDestHandle,
                                  DWORD   dwObjectType,
                                  DWORD&  dwObjectHandle)
{
  int32_t   lRet           = CIFX_NO_ERROR;
  uint8_t*  pbRecvData     = NULL;
  uint32_t  ulRecvDataLen  = 0;
  CF_CREATEINSTANCE_REQ_T tCreateInstance = {0};

  GenerateHeader(&tCreateInstance.tHeader,
                 dwDestHandle,
                 MARSHALLER_CF_METHODID_CREATEINSTANCE,
                 true,
                 sizeof(tCreateInstance.tData));

  tCreateInstance.tData.uHandle.ulHandle = dwObjectType;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tCreateInstance,
                                            sizeof(tCreateInstance),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    PCF_CREATEINSTANCE_CNF_T ptCnf = reinterpret_cast<PCF_CREATEINSTANCE_CNF_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tCreateInstance.tHeader, &ptCnf->tHeader)) )
    {
      dwObjectHandle = ptCnf->tData.uHandle.ulHandle;
    }

    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Delete a channel/sysdevice object on the remote end
///   \param dwObjectHandle Handle of object to destroy
///   \param dwChannelNr    Channel number to object 
///                         (used to differentiate between systemdevice / channel)
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::DeleteDeviceObject(DWORD dwObjectHandle, DWORD dwChannelNr)
{
  int32_t   lRet           = CIFX_NO_ERROR;
  uint8_t*  pbRecvData     = NULL;
  uint32_t  ulRecvDataLen  = 0;

  MARSHALLER_DATA_FRAME_HEADER_T  tCloseObject = {0};

  GenerateHeader(&tCloseObject,
                 dwObjectHandle,
                 (dwChannelNr == CIFX_SYSTEM_DEVICE) ? MARSHALLER_SYSDEV_METHODID_CLOSE : MARSHALLER_CHANNEL_METHODID_CLOSE,
                 true,
                 0);

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tCloseObject,
                                            sizeof(tCloseObject),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    MARSHALLER_DATA_FRAME_HEADER_T* ptCnf = reinterpret_cast<MARSHALLER_DATA_FRAME_HEADER_T*>(pbRecvData);

    lRet = VerifyTransferResult(&tCloseObject, ptCnf);

    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Read out marshaller version on the remote end
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::QueryMarshallerVersion(void)
{
  int32_t                         lRet          = CIFX_NO_ERROR;
  uint8_t*                        pbRecvData    = NULL;
  uint32_t                        ulRecvDataLen = 0;
  MARSHALLER_DATA_FRAME_HEADER_T  tCFVersion    = {0};
  
  GenerateHeader(&tCFVersion, 0, MARSHALLER_CF_METHODID_SERVERVERSION, true, 0);

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tCFVersion,
                                            sizeof(tCFVersion),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_INIT_TIMEOUT);

  if(CIFX_NO_ERROR == lRet)
  {
    PCF_SERVERVERSION_CNF_T ptCnf = reinterpret_cast<PCF_SERVERVERSION_CNF_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tCFVersion, &ptCnf->tHeader)) )
    {
      m_ulVersion = ptCnf->tData.ulVersion;
    }

    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Generate a unique marshaller header 
///   \param  ptHeader     Header to create
///   \param  ulHandle     Destination handle
///   \param  ulMethodId   Method to call on remote object
///   \param  fRequest     true = request, false = response
///   \param  ulDataSize   Size of the data being appended
///   \return void
/////////////////////////////////////////////////////////////////////////////
void CMarshaller::GenerateHeader(PMARSHALLER_DATA_FRAME_HEADER_T  ptHeader,
                                 uint32_t                         ulHandle,
                                 uint32_t                         ulMethodId,
                                 bool                             fRequest,
                                 uint32_t                         ulDataSize)
{
  memset(ptHeader,0,sizeof(*ptHeader));

  ptHeader->ulMethodID                         = ulMethodId;
  ptHeader->uHandle.ulHandle                   = ulHandle;
  ptHeader->ulDataSize                         = ulDataSize;
  ptHeader->uSequence.tBfSequence.fRequest     = fRequest ? 1 : 0;
  
  ptHeader->uSequence.tBfSequence.fSeqNrSupport = m_fSequenceNrSupport ? 1 : 0;
  ptHeader->uSequence.tBfSequence.usSequenceNr = m_fSequenceNrSupport ? ++m_usSequenceNr : 0;
  

}

/////////////////////////////////////////////////////////////////////////////
/// Verify header of result packet
///   \param  ptReq     Header of packet request
///   \param  ptCnf     Header of packet result
///   \return CIFX_NO_ERROR if header is valid
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::VerifyTransferResult ( PMARSHALLER_DATA_FRAME_HEADER_T ptReq,
                                            PMARSHALLER_DATA_FRAME_HEADER_T ptCnf)
{
  int32_t lRet = CIFX_NO_ERROR;

  if ( (ptCnf->ulMethodID       != ptReq->ulMethodID)       ||
       (ptCnf->uHandle.ulHandle != ptReq->uHandle.ulHandle) ||
       (ptCnf->uSequence.tBfSequence.fRequest)                 )
  {
    lRet = CIFX_TRANSPORT_INVALID_RESPONSE;
  } else
  {
    lRet = ptCnf->ulError;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Close driver on remote end
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::CloseDriver(void)
{
  int32_t                         lRet           = CIFX_NO_ERROR;
  uint8_t*                        pbRecvData     = NULL;
  uint32_t                        ulRecvDataLen  = 0;
  MARSHALLER_DATA_FRAME_HEADER_T  tCloseDriver   = {0};

  GenerateHeader(&tCloseDriver, 
                 m_dwDriverHandle, 
                 MARSHALLER_DRV_METHODID_CLOSE, 
                 true, 
                 0);

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tCloseDriver,
                                            sizeof(tCloseDriver),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    MARSHALLER_DATA_FRAME_HEADER_T* ptCnf = reinterpret_cast<MARSHALLER_DATA_FRAME_HEADER_T*>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tCloseDriver, ptCnf)) )
    {
      m_dwDriverHandle = 0;
    }

    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Enumerate boards 
///   \param    ulBoard       Number of Board
///   \param    ulSize        Size of buffer
///   \param    pvBoardInfo   Pointer of buffer (BOARD_INFORMATION)
///   \return   CIFX_NO_ERROR on succes
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::EnumDevice(uint32_t ulBoard, uint32_t ulSize, void* pvBoardInfo)
{

  int32_t                         lRet            = CIFX_NO_ERROR;
  PMARSHALLER_DATA_FRAME_HEADER_T ptHeader        = NULL;
  uint32_t                        ulRequestSize   = 0;

  ulRequestSize = sizeof(*ptHeader)  + 
                  sizeof(DWORD)      +  /* ulBoard   */
                  sizeof(DWORD);        /* ulSize    */
  
  ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);

  memset(ptHeader, 0, ulRequestSize);

  size_t   ulOffset = 0;
  uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptHeader + 1);

  memcpy(&pbData[ulOffset], &ulBoard, sizeof(DWORD));
  ulOffset += sizeof(ulBoard);
  
  memcpy(&pbData[ulOffset], &ulSize, sizeof(DWORD));
  ulOffset += sizeof(ulSize);

  uint8_t*  pbRecvData    = NULL;
  uint32_t  ulRecvDataLen = 0;

  GenerateHeader(ptHeader,
                 m_dwDriverHandle,
                 MARSHALLER_DRV_METHODID_ENUMBOARDS,
                 true,
                 (sizeof(ulBoard) + sizeof(ulSize)));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)ptHeader,
                                            ulRequestSize,
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    PCHANNEL_GETMBXSTATE_CNF_T ptCnf = reinterpret_cast<PCHANNEL_GETMBXSTATE_CNF_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(ptHeader, &ptCnf->tHeader)) )
    {
      memcpy( pvBoardInfo, &ptCnf->tData, ulSize);
    }

    delete [] pbRecvData;
  }

  delete [] ptHeader;

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Enumerate Channels 
///   \param    ulBoard       Number of Board
///   \param    ulChannel     Number of Channel
///   \param    ulSize        Size of buffer
///   \param    pvChannelInfo Pointer of buffer (CHANNEL_INFORMATION)
///   \return   CIFX_NO_ERROR on succes
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::EnumChannel(uint32_t ulBoard, uint32_t ulChannel, uint32_t ulSize, void* pvChannelInfo)
{
  int32_t                         lRet           = CIFX_NO_ERROR;
  PMARSHALLER_DATA_FRAME_HEADER_T ptHeader       = NULL;
  uint32_t                        ulRequestSize  = 0;

  ulRequestSize = sizeof(*ptHeader)  + 
                  sizeof(DWORD)      +  /* ulBoard   */
                  sizeof(DWORD)      +  /* ulChannel   */
                  sizeof(DWORD);        /* ulSize    */

  ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);

  memset(ptHeader, 0, ulRequestSize);

  size_t         ulOffset = 0;
  uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptHeader + 1);

  memcpy(&pbData[ulOffset], &ulBoard, sizeof(DWORD));
  ulOffset += sizeof(ulBoard);

  memcpy(&pbData[ulOffset], &ulChannel, sizeof(DWORD));
  ulOffset += sizeof(ulChannel);

  memcpy(&pbData[ulOffset], &ulSize, sizeof(DWORD));
  ulOffset += sizeof(ulSize);

  uint8_t* pbRecvData    = NULL;
  uint32_t  ulRecvDataLen = 0;

  GenerateHeader(ptHeader, 
                 m_dwDriverHandle,
                 MARSHALLER_DRV_METHODID_ENUMCHANNELS,
                 true,
                 (sizeof(ulBoard) + sizeof(ulChannel) + sizeof(ulSize)));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)ptHeader,
                                            ulRequestSize,
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    PMARSHALLER_DATA_FRAME_HEADER_T ptFrame = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(ptHeader, ptFrame)) )
    {
      CHANNEL_INFORMATION* ptChannelInfo = (CHANNEL_INFORMATION*)(ptFrame + 1);

      memcpy( pvChannelInfo, ptChannelInfo, ulSize);
    }

    delete [] pbRecvData;
  }

  delete [] ptHeader;

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Restart device
///   \param    szBoard       Name of Board
///   \return   CIFX_NO_ERROR on succes
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::RestartDevice(char* szBoard)
{
  int32_t                 lRet          = CIFX_INVALID_HANDLE;
  uint8_t*                pbRecvData    = NULL;
  uint32_t                ulRecvDataLen = 0;
  DRV_RESTARTDEVICE_REQ_T tRestartREQ   = {0};

  strcpy(tRestartREQ.tData.abBoardName, szBoard);

  GenerateHeader(&tRestartREQ.tHeader,
                 m_dwDriverHandle,
                 MARSHALLER_DRV_METHODID_RESTARTDEVICE,
                 true,
                 sizeof(tRestartREQ.tData));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tRestartREQ,
                                            sizeof(tRestartREQ),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    PMARSHALLER_DATA_FRAME_HEADER_T ptCnf = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(pbRecvData);
    lRet = VerifyTransferResult(&tRestartREQ.tHeader, ptCnf);
    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdevice functions

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceOpen, open a Sysdevice 
///   \param hDriver          Handle of driver
///   \param szBoard          Name of the board, which should be open
///   \param phSysdevice      Reference for the new Sysdevice handle 
///   \return CIFX_NO_ERROR,  if success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceOpen( CIFXHANDLE  hDriver, char*   szBoard, CIFXHANDLE* phSysdevice)
{
  UNREFERENCED_PARAMETER(hDriver);
  return OpenChannel( szBoard, CIFX_SYSTEM_DEVICE, phSysdevice);
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceClose, close a Sysdevice, by using handle
///   \param hSysdevice       Handle of sysdevice
///   \return CIFX_NO_ERROR,  if success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceClose( CIFXHANDLE  hSysdevice)
{
  MARSH_CHANNEL_INFO_T* ptChannel = (MARSH_CHANNEL_INFO_T* )hSysdevice;

  int32_t lRet = CIFX_NO_ERROR;

  // Interlocked Decrement 
  lRet = CloseChannel(ptChannel);

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceGetMBXState, get the mail box state of the sysdevice 
///   \param hSysdevice         Handle of sysdevice
///   \param pulRecvPktCount    Reference of the receive packet counter
///   \param pulSendPktCount    Reference of number of sendable packets 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceGetMBXState( CIFXHANDLE  hSysdevice, uint32_t* pulRecvPktCount, uint32_t* pulSendPktCount)
{
  PMARSH_CHANNEL_INFO_T ptSysDevice = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hSysdevice);

  return GetMbxState( ptSysDevice, pulRecvPktCount, pulSendPktCount);
}


/////////////////////////////////////////////////////////////////////////////
/// xSysdevicePutPacket, put a sendable packet in send queue
///   \param hSysdevice         Handle of sysdevice
///   \param ptSendPkt          Reference of sendable packet 
///   \param ulTimeout          Time out for the send process
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdevicePutPacket( CIFXHANDLE  hSysdevice, CIFX_PACKET* ptSendPkt, uint32_t ulTimeout)
{
  PMARSH_CHANNEL_INFO_T ptSysDevice = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hSysdevice);
	
  return PutPacket( ptSysDevice, ptSendPkt, ulTimeout);
}
                              
/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceGetPacket, get a received packet from the recevice queue
///   \param hSysdevice         Handle of sysdevice
///   \param ulSize             Size of the receive date buffer
///   \param ptRecvPkt          Reference for the received packet data
///   \param ulTimeout          Time out of the receive packet proecess
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceGetPacket( CIFXHANDLE  hSysdevice, uint32_t ulSize, CIFX_PACKET* ptRecvPkt, uint32_t ulTimeout)
{
  PMARSH_CHANNEL_INFO_T ptSysDevice = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hSysdevice);

  return GetPacket( ptSysDevice, ptRecvPkt, ulSize, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceInfo, get special information from the device (Channel)
///   \param hSysdevice         Handle of sysdevice
///   \param ulCmd              Command, which special informations are ordered
///   \param ulSize             Size of the data buffer for the information structure
///   \param pvInfo             Reference (void), for the orderd data 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceInfo( CIFXHANDLE  hSysdevice, uint32_t ulCmd, uint32_t ulSize, void* pvInfo)
{
  // Get Marshaller channel handling structure, if handle is valid 
  PMARSH_CHANNEL_INFO_T ptSysDevice = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hSysdevice);

  return  (int32_t) MethodeInfo( ptSysDevice, ulCmd, ulSize, pvInfo, CHANNEL_TYPE_XSYSDEVICE);
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceFindFirstFile, 
///   \param hSysdevice         Handle of system device 
///   \param ulChannel          Number of channel
///   \param ptDirectoryInfo    Reference of the directory information
///   \param pfnRecvPktCallback Callback for the received packets
///   \param pvUser             User pointer
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceFindFirstFile( CIFXHANDLE            hSysdevice,
                                              uint32_t              ulChannel,
                                              CIFX_DIRECTORYENTRY*  ptDirectoryInfo,
                                              PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                              void*                 pvUser)
{
  int32_t               lRet          = CIFX_INVALID_HANDLE;
  uint8_t*              pbRecvData    = NULL;
  uint32_t              ulRecvDataLen = 0;
  PMARSH_CHANNEL_INFO_T ptSysDevice   = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hSysdevice);

  if (ptSysDevice->ulChannelNr == CIFX_SYSTEM_DEVICE)
  {
    SYSDEV_FIND_FIRSTFILE_REQ_T  tFindFirst = {0};

    GenerateHeader( &tFindFirst.tHeader, 
                    ptSysDevice->dwRemoteHandle,
                    MARSHALLER_SYSDEV_METHODID_FINDFIRSTFILE,
                    true,
                    sizeof(tFindFirst.tData));

    // NOTE: This implementation works only on 32bit machines 
    tFindFirst.tData.hList              = (DWORD) ptDirectoryInfo->hList;
    tFindFirst.tData.ulFilesize         = ptDirectoryInfo->ulFilesize;
    tFindFirst.tData.bFiletype          = ptDirectoryInfo->bFiletype;
    tFindFirst.tData.ulChannel          = ulChannel;
    tFindFirst.tData.ulRecvPktCallback  = (DWORD) pfnRecvPktCallback;
    tFindFirst.tData.ulUser             = (DWORD) pvUser;
    memcpy( &tFindFirst.tData.szFilename, &ptDirectoryInfo->szFilename, sizeof(ptDirectoryInfo->szFilename));

    lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                              (uint8_t*)&tFindFirst,
                                              sizeof(tFindFirst),
                                              pbRecvData,
                                              ulRecvDataLen,
                                              m_pcEndpoint,
                                              0,
                                              CIFX_TO_CONT_PACKET);
    if(CIFX_NO_ERROR == lRet)
    {
      PSYSDEV_FIND_FIRSTFILE_CNF_T ptCnf = reinterpret_cast<PSYSDEV_FIND_FIRSTFILE_CNF_T>(pbRecvData);
      if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tFindFirst.tHeader, &ptCnf->tHeader)) )
      {
        // Note: this implementation works only on 32 bit machines */
        ptDirectoryInfo->hList        = (CIFXHANDLE) ptCnf->tData.hList;
        ptDirectoryInfo->ulFilesize   = ptCnf->tData.ulFilesize;
        ptDirectoryInfo->bFiletype    = ptCnf->tData.bFiletype ;
        memset( &ptDirectoryInfo->szFilename, 0, sizeof(ptDirectoryInfo->szFilename));
        memcpy( &ptDirectoryInfo->szFilename, &ptCnf->tData.szFilename, sizeof(ptDirectoryInfo->szFilename));
      }
      delete [] pbRecvData;
    }

  } else
  {
    CHANNEL_FIND_FIRSTFILE_REQ_T  tFindFirst = {0};

    GenerateHeader( &tFindFirst.tHeader,
                    ptSysDevice->dwRemoteHandle,
                    MARSHALLER_CHANNEL_METHODID_FINDFIRSTFILE,
                    true,
                    sizeof(tFindFirst.tData));

    // NOTE: This implementation works only on 32bit machines 
    tFindFirst.tData.hList              = (DWORD) ptDirectoryInfo->hList;
    tFindFirst.tData.ulFilesize         = ptDirectoryInfo->ulFilesize;
    tFindFirst.tData.bFiletype          = ptDirectoryInfo->bFiletype;
    tFindFirst.tData.ulRecvPktCallback  = (DWORD) pfnRecvPktCallback;
    tFindFirst.tData.ulUser             = (DWORD) pvUser;
    memcpy( &tFindFirst.tData.szFilename, &ptDirectoryInfo->szFilename, sizeof(ptDirectoryInfo->szFilename));

    lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                              (uint8_t*)&tFindFirst,
                                              sizeof(tFindFirst),
                                              pbRecvData,
                                              ulRecvDataLen,
                                              m_pcEndpoint,
                                              0,
                                              CIFX_TO_CONT_PACKET);

    if(CIFX_NO_ERROR == lRet)
    {
      PCHANNEL_FIND_FIRSTFILE_CNF_T ptCnf = reinterpret_cast<PCHANNEL_FIND_FIRSTFILE_CNF_T>(pbRecvData);
      if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tFindFirst.tHeader, &ptCnf->tHeader)) )
      {
        // Note: this implementation works only on 32 bit machines */
        ptDirectoryInfo->hList        = (CIFXHANDLE) ptCnf->tData.hList;
        ptDirectoryInfo->ulFilesize   = ptCnf->tData.ulFilesize;
        ptDirectoryInfo->bFiletype    = ptCnf->tData.bFiletype ;
        memset( &ptDirectoryInfo->szFilename, 0, sizeof(ptDirectoryInfo->szFilename));
        memcpy( &ptDirectoryInfo->szFilename, &ptCnf->tData.szFilename, sizeof(ptDirectoryInfo->szFilename));
      }
      delete [] pbRecvData;
    }
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceFindNextFile,
///   \param hSysdevice         Handle of system device 
///   \param ulChannel          Number of channel
///   \param ptDirectoryInfo    Reference of the directory information
///   \param pfnRecvPktCallback Callback for the received packets
///   \param pvUser             User pointer
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceFindNextFile(CIFXHANDLE            hSysdevice,
                                            uint32_t              ulChannel,
                                            CIFX_DIRECTORYENTRY*  ptDirectoryInfo,
                                            PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                            void*                 pvUser)
{
  int32_t               lRet             = CIFX_INVALID_HANDLE;
  uint8_t*              pbRecvData       = NULL;
  uint32_t              ulRecvDataLen    = 0;
  PMARSH_CHANNEL_INFO_T ptSysDevice      = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hSysdevice);

  if (ptSysDevice->ulChannelNr == CIFX_SYSTEM_DEVICE)
  {
    SYSDEV_FIND_NEXTFILE_REQ_T  tFindNext     = {0};

    GenerateHeader( &tFindNext.tHeader, 
                    ptSysDevice->dwRemoteHandle,
                    MARSHALLER_SYSDEV_METHODID_FINDNEXTFILE,
                    true,
                    sizeof(tFindNext.tData));

    // NOTE: This implementation works only on 32bit machines 
    tFindNext.tData.hList              = (DWORD) ptDirectoryInfo->hList;
    tFindNext.tData.ulFilesize         = ptDirectoryInfo->ulFilesize;
    tFindNext.tData.bFiletype          = ptDirectoryInfo->bFiletype;
    tFindNext.tData.ulChannel          = ulChannel;
    tFindNext.tData.ulRecvPktCallback  = (DWORD) pfnRecvPktCallback;
    tFindNext.tData.ulUser             = (DWORD) pvUser;
    memcpy( &tFindNext.tData.szFilename, &ptDirectoryInfo->szFilename, sizeof(ptDirectoryInfo->szFilename));   

    lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                              (uint8_t*)&tFindNext,
                                              sizeof(tFindNext),
                                              pbRecvData,
                                              ulRecvDataLen,
                                              m_pcEndpoint,
                                              0,
                                              CIFX_TO_CONT_PACKET);
    if(CIFX_NO_ERROR == lRet)
    {
      PSYSDEV_FIND_NEXTFILE_CNF_T ptCnf = reinterpret_cast<PSYSDEV_FIND_NEXTFILE_CNF_T>(pbRecvData);
      if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tFindNext.tHeader, &ptCnf->tHeader)) )
      {
        // Note: this implementation works only on 32 bit machines */
        ptDirectoryInfo->hList        = (CIFXHANDLE) ptCnf->tData.hList;
        ptDirectoryInfo->ulFilesize   = ptCnf->tData.ulFilesize;
        ptDirectoryInfo->bFiletype    = ptCnf->tData.bFiletype ;
        memset( &ptDirectoryInfo->szFilename, 0, sizeof(ptDirectoryInfo->szFilename));
        memcpy( &ptDirectoryInfo->szFilename, &ptCnf->tData.szFilename, sizeof(ptDirectoryInfo->szFilename));
      }
      delete [] pbRecvData;
    }

  } else
  {
    CHANNEL_FIND_NEXTFILE_REQ_T tFindNext = {0};

    GenerateHeader( &tFindNext.tHeader, 
                    ptSysDevice->dwRemoteHandle, 
                    MARSHALLER_CHANNEL_METHODID_FINDNEXTFILE,
                    true, 
                    sizeof(tFindNext.tData));

    // NOTE: This implementation works only on 32bit machines 
    tFindNext.tData.hList              = (DWORD) ptDirectoryInfo->hList;
    tFindNext.tData.ulFilesize         = ptDirectoryInfo->ulFilesize;
    tFindNext.tData.bFiletype          = ptDirectoryInfo->bFiletype;
    tFindNext.tData.ulRecvPktCallback  = (DWORD) pfnRecvPktCallback;
    tFindNext.tData.ulUser             = (DWORD) pvUser;
    memcpy( &tFindNext.tData.szFilename, &ptDirectoryInfo->szFilename, sizeof(ptDirectoryInfo->szFilename));   

    lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                              (uint8_t*)&tFindNext,
                                              sizeof(tFindNext),
                                              pbRecvData,
                                              ulRecvDataLen,
                                              m_pcEndpoint,
                                              0,
                                              CIFX_TO_CONT_PACKET);

    if(CIFX_NO_ERROR == lRet)
    { 
      PCHANNEL_FIND_NEXTFILE_CNF_T ptCnf = reinterpret_cast<PCHANNEL_FIND_NEXTFILE_CNF_T>(pbRecvData);
        
      if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tFindNext.tHeader, &ptCnf->tHeader)) )
      {
        // Note: this implementation works only on 32 bit machines */
        ptDirectoryInfo->hList        = (CIFXHANDLE) ptCnf->tData.hList;
        ptDirectoryInfo->ulFilesize   = ptCnf->tData.ulFilesize;
        ptDirectoryInfo->bFiletype    = ptCnf->tData.bFiletype ;
        memset( &ptDirectoryInfo->szFilename, 0, sizeof(ptDirectoryInfo->szFilename));
        memcpy( &ptDirectoryInfo->szFilename, &ptCnf->tData.szFilename, sizeof(ptDirectoryInfo->szFilename)); 
      }
      delete [] pbRecvData;
    }
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceDownload, 
///   \param hSysdevice         Handle of system device 
///   \param ulChannel          Number of channel
///   \param ulMode             Mode of Download (How to handle download)
///   \param szFileName         Reference of the file name 
///   \param pabFileData        Reference of file data 
///   \param ulFileSize         Size of file 
///   \param pfnCallback        Callback for progress 
///   \param pfnRecvPktCallback Callback for the recevied packets 
///   \param pvUser             User pointer
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceDownload(CIFXHANDLE            hSysdevice,
                                        uint32_t              ulChannel,
                                        uint32_t              ulMode,
                                        char*                 szFileName,
                                        uint8_t*              pabFileData,
                                        uint32_t              ulFileSize,
                                        PFN_PROGRESS_CALLBACK pfnCallback,
                                        PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                        void*                 pvUser)
{
  int32_t               lRet        = CIFX_NO_ERROR;
  PMARSH_CHANNEL_INFO_T ptSysDevice = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hSysdevice);

  if(ptSysDevice->lRefCount > 0)
  {
    uint32_t  ulTransferType = 0;

    switch(ulMode)
    {
    case DOWNLOAD_MODE_FIRMWARE:
      if( CIFX_NO_ERROR != (lRet = DEV_GetFWTransferTypeFromFileName( szFileName, &ulTransferType)))
        return lRet;

      DEV_RemoveChannelFiles( hSysdevice,
                              ulChannel,
                              this,
                              pfnRecvPktCallback,
                              pvUser,
                              NULL);
      break;

    case DOWNLOAD_MODE_CONFIG:
    case DOWNLOAD_MODE_FILE:
      ulTransferType = RCX_FILE_XFER_FILE;
      break;

    case DOWNLOAD_MODE_LICENSECODE:
      ulTransferType = RCX_FILE_XFER_LICENSE_CODE;
      break;

    default:
      return CIFX_INVALID_PARAMETER;
    }

    lRet = DEV_DownloadFile(hSysdevice,
                            ulChannel,
                            CIFX_MAX_PACKET_SIZE,
                            ulTransferType,
                            szFileName,
                            ulFileSize,
                            pabFileData,
                            this,
                            pfnCallback,
                            pfnRecvPktCallback,
                            pvUser);
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceUpload, 
///   \param hSysdevice         Handle of system device 
///   \param ulChannel          Number of channel
///   \param ulMode             Mode of Download (How to handle download)
///   \param szFileName         Reference of the file name 
///   \param pabFileData        Reference of file data 
///   \param pulFileSize        Length of the provided buffer, returned length of data
///   \param pfnCallback        Callback for progress 
///   \param pfnRecvPktCallback Callback for the recevied packets 
///   \param pvUser             User pointer
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceUpload(CIFXHANDLE            hSysdevice,
                                      uint32_t              ulChannel,
                                      uint32_t              ulMode,
                                      char*                 szFileName,
                                      uint8_t*              pabFileData,
                                      uint32_t*             pulFileSize,
                                      PFN_PROGRESS_CALLBACK pfnCallback,
                                      PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                      void*                 pvUser)
{
  // NOTE: We ignore the RecvCallback functions pointer, cause we don't receive a async packet

  int32_t               lRet        = CIFX_NO_ERROR;
  PMARSH_CHANNEL_INFO_T ptSysDevice = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hSysdevice);

  if(ptSysDevice->lRefCount > 0)
  {
    uint32_t   ulTransferType  = 0;

    switch(ulMode)
    {
    case DOWNLOAD_MODE_FIRMWARE:
      if( CIFX_NO_ERROR != (lRet = DEV_GetFWTransferTypeFromFileName( szFileName, &ulTransferType)))
        return lRet;
      break;

    case DOWNLOAD_MODE_CONFIG:
    case DOWNLOAD_MODE_FILE:
      ulTransferType = RCX_FILE_XFER_FILE;
      break;

    default:
      return CIFX_INVALID_PARAMETER;
    }

    lRet =  DEV_UploadFile( hSysdevice,
                            ulChannel,
                            CIFX_MAX_PACKET_SIZE,
                            ulTransferType,
                            szFileName,
                            pulFileSize,
                            pabFileData,
                            this,
                            pfnCallback,
                            pfnRecvPktCallback,
                            pvUser);


  }else
  {
    lRet = CIFX_INVALID_HANDLE;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xSysdeviceReset, Signal reset to the Sysdevice  
///   \param  hSysdevice          Reference to the channel structure
///   \param  ulTimeout          Timeout in ms 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xSysdeviceReset( CIFXHANDLE  hSysdevice,
                                      uint32_t    ulTimeout)
{
  int32_t lRet = CIFX_NO_ERROR;

  // Get Marshaller channel handling structure, if handle is valid 
  PMARSH_CHANNEL_INFO_T ptSysDevice = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hSysdevice);

  lRet = Reset( ptSysDevice, NULL, CHANNEL_TYPE_XSYSDEVICE,  ulTimeout);

  /* Reconnect will be handled by calling layer */

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannel functions


/////////////////////////////////////////////////////////////////////////////
/// xChannelOpen,
///   \param  hDriver        Handle of the driver 
///   \param  szBoard        Name of the device 
///   \param  ulChannel      Number of channel 
///   \param  phChannel      Reference of the channel handle 
///   \return CIFX_NO_ERROR onsuccess
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelOpen( CIFXHANDLE  hDriver,  char* szBoard, uint32_t ulChannel, CIFXHANDLE* phChannel)
{
  UNREFERENCED_PARAMETER(hDriver);
  return OpenChannel(szBoard, ulChannel, phChannel);
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelClose,
///   \param  hChannel        Handle of a channel
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelClose( CIFXHANDLE  hChannel)
{
  MARSH_CHANNEL_INFO_T* ptChannel = (MARSH_CHANNEL_INFO_T* )hChannel;

  int32_t lRet = CIFX_NO_ERROR; 

  // Interlocked Decrement 
  lRet = CloseChannel(ptChannel);

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelFindFirstFile,
///   \param  hChannel            Handle of a channel
///   \param  ptDirectoryInfo     Reference of the directory info structure
///   \param  pfnRecvPktCallback  Callback for the receveived packets 
///   \param  pvUser              User pointer 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelFindFirstFile( CIFXHANDLE            hChannel,
                                            CIFX_DIRECTORYENTRY*  ptDirectoryInfo,
                                            PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                            void*                 pvUser)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  if(ptChannel->lRefCount > 0)
  {
    lRet = xSysdeviceFindFirstFile( hChannel,
                                    ptChannel->ulChannelNr,
                                    ptDirectoryInfo,
                                    pfnRecvPktCallback,
                                    pvUser);
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelFindNextFile,
///   \param  hChannel            Handle of a channel
///   \param  ptDirectoryInfo     Reference of the directory info structure
///   \param  pfnRecvPktCallback  Callback for the receveived packets 
///   \param  pvUser              User pointer 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelFindNextFile(CIFXHANDLE            hChannel,
                                          CIFX_DIRECTORYENTRY*  ptDirectoryInfo,
                                          PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                          void*                 pvUser)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  if(ptChannel->lRefCount > 0)
  {
    lRet = xSysdeviceFindNextFile( hChannel,
                                   ptChannel->ulChannelNr,
                                   ptDirectoryInfo,
                                   pfnRecvPktCallback,
                                   pvUser);
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelDownload,
///   \param  hChannel            Handle of a channel
///   \param  ulMode              Mode of download (how to handle download)
///   \param  szFileName          Reference of the filename
///   \param  pabFileData         Data pointer
///   \param  ulFileSize          Size of file 
///   \param  pfnCallback         Callback for the progress
///   \param  pfnRecvPktCallback  callback for received packets 
///   \param  pvUser              User pointer 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelDownload(CIFXHANDLE            hChannel,
                                      uint32_t              ulMode,
                                      char*                 szFileName,
                                      uint8_t*              pabFileData,
                                      uint32_t              ulFileSize,
                                      PFN_PROGRESS_CALLBACK pfnCallback,
                                      PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                      void*                 pvUser)
{
  int32_t               lRet      = CIFX_NO_ERROR;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  if(ptChannel->lRefCount > 0)
  {

    lRet = xSysdeviceDownload(hChannel,
                              ptChannel->ulChannelNr,
                              ulMode,
                              szFileName, 
                              pabFileData,
                              ulFileSize,
                              pfnCallback,
                              pfnRecvPktCallback,
                              pvUser);
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelUpload,
///   \param  hChannel            Handle of a channel
///   \param  ulMode              Mode of download (how to handle download)
///   \param  szFileName          Reference of the filename
///   \param  pabFileData         Data pointer
///   \param  pulFileSize         Length of the provided buffer, returned length of data
///   \param  pfnCallback         Callback for the progress
///   \param  pfnRecvPktCallback  callback for received packets 
///   \param  pvUser              User pointer 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelUpload(CIFXHANDLE            hChannel,
                                    uint32_t              ulMode,
                                    char*                 szFileName,
                                    uint8_t*              pabFileData,
                                    uint32_t*             pulFileSize,
                                    PFN_PROGRESS_CALLBACK pfnCallback,
                                    PFN_RECV_PKT_CALLBACK pfnRecvPktCallback,
                                    void*                 pvUser)
{
  int32_t               lRet      = CIFX_NO_ERROR;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  if(ptChannel->lRefCount > 0)
  {

    lRet = xSysdeviceUpload(hChannel,
                            ptChannel->ulChannelNr,
                            ulMode,
                            szFileName,
                            pabFileData,
                            pulFileSize,
                            pfnCallback,
                            pfnRecvPktCallback,
                            pvUser);
  }else
  {
    lRet = CIFX_INVALID_HANDLE;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelGetMBXState,
///   \param  hChannel            Handle of a channel
///   \param  pulRecvPktCount     Reference of the received packet counter
///   \param  pulSendPktCount     Reference of the send packet counter
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelGetMBXState( CIFXHANDLE  hChannel, uint32_t* pulRecvPktCount, uint32_t* pulSendPktCount)
{
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  return GetMbxState(ptChannel, pulRecvPktCount, pulSendPktCount);
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelPutPacket, put a packet into send queue (send a packet)
///   \param  hChannel      Handle for a channel (reference of a managment channel structure)
///   \param  ptSendPkt    Reference of the send packet 
///   \param  ulTimeout    Timeount value for the send routine
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelPutPacket( CIFXHANDLE  hChannel, CIFX_PACKET*  ptSendPkt, uint32_t ulTimeout)
{
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  return PutPacket(ptChannel, ptSendPkt, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelGetPacket, Get a received packet
///   \param  hChannel    Handle of the managment channel structure
///   \param  ulSize      Size of receive buffer for the received packet 
///   \param  ptRecvPkt   Reference for a received packet 
///   \param  ulTimeout   Timeout for the get packet process
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelGetPacket( CIFXHANDLE  hChannel, uint32_t ulSize, CIFX_PACKET* ptRecvPkt, uint32_t ulTimeout)
{

  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  return GetPacket(ptChannel, ptRecvPkt, ulSize, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelGetSendPacket, Get the last send packet 
///   \param  hChannel    Handle of the managment channel structure
///   \param  ulSize      Size of receive buffer for the received packet 
///   \param  ptRecvPkt   Reference for a received packet 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelGetSendPacket( CIFXHANDLE    hChannel,
                                            uint32_t      ulSize,
                                            CIFX_PACKET*  ptRecvPkt)
{
  int32_t       lRet          = CIFX_INVALID_HANDLE;
  uint8_t*      pbRecvData    = NULL;
  uint32_t      ulRecvDataLen = 0;

  CHANNEL_GET_SENDPACKET_REQ_T  tGetSendPkt = {0};

  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  GenerateHeader( &tGetSendPkt.tHeader, 
                  ptChannel->dwRemoteHandle, 
                  MARSHALLER_CHANNEL_METHODID_GETSENDPACKET,
                  true, 
                  sizeof(tGetSendPkt.tData));

  tGetSendPkt.tData.ulSize    = ulSize;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tGetSendPkt,
                                            sizeof(tGetSendPkt),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    PMARSHALLER_DATA_FRAME_HEADER_T ptMarshHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(pbRecvData);
    CIFX_PACKET*                    ptPacket      = reinterpret_cast<CIFX_PACKET*>(ptMarshHeader + 1);

    uint32_t ulCopySize = ptMarshHeader->ulDataSize;

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tGetSendPkt.tHeader, ptMarshHeader)) )
    {
      if(ulSize < ulCopySize)
      {
        ulCopySize = ulSize;
        lRet = CIFX_BUFFER_TOO_SHORT;
      }

      memcpy(ptRecvPkt, ptPacket, ulCopySize);
    }
    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelConfigLock, lock configuration 
///   \param  hChannel    Handle of the managment channel structure
///   \param  ulCmd       Command for the lock functionality 
///   \param  pulState    Reference for the State
///   \param  ulTimeout   Timeout for the function call 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelConfigLock(CIFXHANDLE  hChannel,
                                        uint32_t    ulCmd,
                                        uint32_t*   pulState,
                                        uint32_t    ulTimeout)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  uint8_t*                  pbRecvData      = NULL;
  uint32_t                  ulRecvDataLen   = 0;
  CHANNEL_CONFIG_LOCK_REQ_T tConfigLockREQ  = {0};

  tConfigLockREQ.tData.ulCmd      = ulCmd;
  tConfigLockREQ.tData.ulState    = *pulState;
  tConfigLockREQ.tData.ulTimeout  = ulTimeout;

  GenerateHeader( &tConfigLockREQ.tHeader,
                  ptChannel->dwRemoteHandle,
                  MARSHALLER_CHANNEL_METHODID_CONFIGLOCK,
                  true,
                  sizeof(tConfigLockREQ.tData));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tConfigLockREQ,
                                            sizeof(tConfigLockREQ),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            ulTimeout);

  if(CIFX_NO_ERROR == lRet)
  {
     PCHANNEL_CONFIG_LOCK_CNF_T ptCnf = reinterpret_cast<PCHANNEL_CONFIG_LOCK_CNF_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tConfigLockREQ.tHeader, &ptCnf->tHeader)) )
    {
      *pulState = ptCnf->tData.ulState;
    }

    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelInfo,
///   \param  hChannel        Handle of a channel
///   \param  ulResetMode     Special mode of reset 
///   \param  ulTimeout       Timeout in ms
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelReset( CIFXHANDLE  hChannel,
                                    uint32_t    ulResetMode,
                                    uint32_t    ulTimeout)
{
  int32_t lRet = CIFX_NO_ERROR;

  // Get Marshaller channel handling structure, if handle is valid 
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);
  
  lRet = Reset( ptChannel, ulResetMode, CHANNEL_TYPE_CHANNEL, ulTimeout);

  /* Reconnect will be handled by calling layer */

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelInfo,
///   \param  hChannel        Handle of a channel
///   \param  ulSize          Size of reveive data buffer 
///   \param  pvChannelInfo   Reference to the receive data buffer 
///   \return CIFX_NO_ERROR,  if success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelInfo( CIFXHANDLE  hChannel, uint32_t ulSize, void* pvChannelInfo)
{
  if(ulSize < sizeof(CHANNEL_INFORMATION))
    return CIFX_INVALID_BUFFERSIZE;

  int32_t lRet = CIFX_NO_ERROR;

  // Get Marshaller channel handling structure, if handle is valid 
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  // Use MethodInfo, by signaling a channel
  lRet = (int32_t) MethodeInfo( ptChannel, NULL, ulSize, pvChannelInfo, CHANNEL_TYPE_CHANNEL);
  
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelWatchdog,
///   \param  hChannel        Handle of a channel
///   \param  ulCmd           Command for the Watchdog 
///   \param  pulTrigger    Reference of the Trigger counter 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelWatchdog(CIFXHANDLE  hChannel,
                                      uint32_t    ulCmd,
                                      uint32_t*   pulTrigger)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  uint8_t*                pbRecvData    = NULL;
  uint32_t                ulRecvDataLen = 0;
  CHANNEL_WATCHDOG_REQ_T  tWatchdogREQ  = {0};

  tWatchdogREQ.tData.ulCmd      = ulCmd;
  tWatchdogREQ.tData.ulTrigger  = *pulTrigger;
  

  GenerateHeader( &tWatchdogREQ.tHeader,
                  ptChannel->dwRemoteHandle, 
                  MARSHALLER_CHANNEL_METHODID_WATCHDOG,
                  true, 
                  sizeof(tWatchdogREQ.tData));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tWatchdogREQ,
                                            sizeof(tWatchdogREQ),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
     PCHANNEL_WATCHDOG_CNF_T ptCnf = reinterpret_cast<PCHANNEL_WATCHDOG_CNF_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tWatchdogREQ.tHeader, &ptCnf->tHeader)) )
    {
      *pulTrigger = ptCnf->tData.ulTrigger;
    }

    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelHostState,
///   \param  hChannel        Handle of a channel
///   \param  ulCmd           Command
///   \param  pulState        Reference for the host state 
///   \param  ulTimeout       Timeout for the function call
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelHostState( CIFXHANDLE  hChannel,
                                        uint32_t    ulCmd,
                                        uint32_t*   pulState,
                                        uint32_t    ulTimeout)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  uint8_t*                  pbRecvData    = NULL;
  uint32_t                  ulRecvDataLen = 0;
  CHANNEL_HOSTSTATE_REQ_T   tHostStateREQ = {0};

  tHostStateREQ.tData.ulCmd     = ulCmd;
  tHostStateREQ.tData.ulTimeout = ulTimeout;
  if(NULL != pulState)
    tHostStateREQ.tData.ulState = *pulState;
  

  GenerateHeader( &tHostStateREQ.tHeader,
                  ptChannel->dwRemoteHandle, 
                  MARSHALLER_CHANNEL_METHODID_HOSTSTATE,
                  true, 
                  sizeof(tHostStateREQ.tData));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tHostStateREQ,
                                            sizeof(tHostStateREQ),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            ulTimeout);

  if(CIFX_NO_ERROR == lRet)
  {
     PCHANNEL_HOSTSTATE_CNF_T ptCnf = reinterpret_cast<PCHANNEL_HOSTSTATE_CNF_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tHostStateREQ.tHeader, &ptCnf->tHeader)) )
    {
      if(NULL != pulState)
      *pulState = ptCnf->tData.ulState;
    }

    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelBusState,
///   \param  hChannel        Handle of a channel
///   \param  ulCmd           Command
///   \param  pulState        Reference for the bus state 
///   \param  ulTimeout       Timeout for the function call
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelBusState(CIFXHANDLE  hChannel,
                                      uint32_t    ulCmd,
                                      uint32_t*   pulState,
                                      uint32_t    ulTimeout)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  uint8_t*                pbRecvData    = NULL;
  uint32_t                ulRecvDataLen = 0;
  CHANNEL_BUSSTATE_REQ_T  tBusStateREQ  = {0};

  tBusStateREQ.tData.ulCmd      = ulCmd;
  tBusStateREQ.tData.ulTimeout  = ulTimeout;
  if(NULL != pulState)
    tBusStateREQ.tData.ulState = *pulState;


  GenerateHeader( &tBusStateREQ.tHeader,
                  ptChannel->dwRemoteHandle,
                  MARSHALLER_CHANNEL_METHODID_BUSSTATE,
                  true,
                  sizeof(tBusStateREQ.tData));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tBusStateREQ,
                                            sizeof(tBusStateREQ),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            ulTimeout);

  if(CIFX_NO_ERROR == lRet)
  {
     PCHANNEL_BUSSTATE_CNF_T ptCnf = reinterpret_cast<PCHANNEL_BUSSTATE_CNF_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tBusStateREQ.tHeader, &ptCnf->tHeader)) )
    {
      if(NULL != pulState)
      *pulState = ptCnf->tData.ulState;
    }

    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelIOInfo,
///   \param  hChannel        Handle of a channel
///   \param  ulCmd           Command
///   \param  ulAreaNumber    Number area 
///   \param  ulSize          Size of buffer for the IO-Information
///   \param  pvData          Reference for the IO-Information
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelIOInfo(CIFXHANDLE  hChannel,
                                    uint32_t    ulCmd,
                                    uint32_t    ulAreaNumber,
                                    uint32_t    ulSize,
                                    void*       pvData)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  CHANNEL_IOINFO_REQ_T  tIOInfoREQ      = {0};
  uint8_t*              pbRecvData      = NULL;
  uint32_t              ulRecvDataLen   = 0;

  tIOInfoREQ.tData.ulCmd      = ulCmd;
  tIOInfoREQ.tData.ulArea     = ulAreaNumber;
  tIOInfoREQ.tData.ulDataLen  = ulSize;

  GenerateHeader( &tIOInfoREQ.tHeader,
                  ptChannel->dwRemoteHandle,
                  MARSHALLER_CHANNEL_METHODID_IOINFO,
                  true,
                  sizeof(tIOInfoREQ.tData));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tIOInfoREQ,
                                            sizeof(tIOInfoREQ),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    PMARSHALLER_DATA_FRAME_HEADER_T ptMarshHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tIOInfoREQ.tHeader, ptMarshHeader)) )
    {
      /* Check the user buffer size and copy at least the length of the user buffer */
      uint32_t ulCopySize = ptMarshHeader->ulDataSize;
      if(ulSize < ulCopySize)
      {
        ulCopySize = ulSize;
        lRet       = CIFX_BUFFER_TOO_SHORT;
      }
      memcpy( pvData, ptMarshHeader + 1, ulCopySize);
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelIORead,
///   \param  hChannel        Handle of a channel
///   \param  ulAreaNumber    Number area 
///   \param  ulOffset        Data start offset 
///   \param  ulDataLen       Data length to read 
///   \param  pvData          Reference for the readed data 
///   \param  ulTimeout       Timeout for the function call
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelIORead( CIFXHANDLE hChannel,
                                     uint32_t   ulAreaNumber,
                                     uint32_t   ulOffset,
                                     uint32_t   ulDataLen,
                                     void*      pvData,
                                     uint32_t   ulTimeout)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  CHANNEL_IOREAD_REQ_T  tIOReadREQ      = {0};
  uint8_t*              pbRecvData      = NULL;
  uint32_t              ulRecvDataLen   = 0;

  tIOReadREQ.tData.ulArea     = ulAreaNumber;
  tIOReadREQ.tData.ulOffset   = ulOffset;
  tIOReadREQ.tData.ulDataLen  = ulDataLen;
  tIOReadREQ.tData.ulTimeout  = ulTimeout;

  GenerateHeader( &tIOReadREQ.tHeader,
                  ptChannel->dwRemoteHandle,
                  MARSHALLER_CHANNEL_METHODID_IOREAD,
                  true,
                  sizeof(tIOReadREQ.tData));


  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tIOReadREQ,
                                            sizeof(tIOReadREQ),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            ulTimeout);

  if(CIFX_NO_ERROR == lRet)
  {
    PMARSHALLER_DATA_FRAME_HEADER_T ptMarshHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(pbRecvData);

    lRet = VerifyTransferResult(&tIOReadREQ.tHeader, ptMarshHeader);

    if ( (CIFX_NO_ERROR == lRet) ||
         (CIFX_DEV_NO_COM_FLAG == lRet) )
    {
      /* Check the user buffer size and copy at least the length of the user buffer */
      uint32_t ulCopySize = ptMarshHeader->ulDataSize;
      if(ulDataLen < ulCopySize)
      {
        ulCopySize = ulDataLen;
        lRet       = CIFX_BUFFER_TOO_SHORT;
      }
      memcpy( pvData, ptMarshHeader + 1, ulCopySize);
    }
    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelIOWrite,
///   \param  hChannel        Handle of a channel
///   \param  ulAreaNumber    Number area 
///   \param  ulOffset        Data start offset 
///   \param  ulDataLen       Data length to write 
///   \param  pvData          Reference for the readed data 
///   \param  ulTimeout       Timeout for the function call
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelIOWrite( CIFXHANDLE  hChannel,
                                      uint32_t    ulAreaNumber,
                                      uint32_t    ulOffset,
                                      uint32_t    ulDataLen,
                                      void*       pvData,
                                      uint32_t    ulTimeout)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  PMARSHALLER_DATA_FRAME_HEADER_T ptHeader      = NULL;
  uint32_t                        ulRequestSize = 0;


  ulRequestSize = sizeof(*ptHeader)     +   /* Transport Header     */
                  sizeof(ulAreaNumber)  +   /* Area number          */
                  sizeof(ulOffset)      +   /* Offset               */
                  sizeof(ulTimeout)     +   /* Timeout in ms        */
                  sizeof(ulDataLen)     +   /* Length of data       */
                  ulDataLen;


  ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);
  memset(ptHeader, 0, ulRequestSize);

  GenerateHeader( ptHeader,
                  ptChannel->dwRemoteHandle, 
                  MARSHALLER_CHANNEL_METHODID_IOWRITE,
                  true,
                  ulRequestSize - sizeof(*ptHeader));


  size_t   ulSendOffset = 0;
  uint8_t* pbData       = reinterpret_cast<uint8_t*>(ptHeader + 1);

  memcpy(&pbData[ulSendOffset], &ulAreaNumber, sizeof(ulAreaNumber));
  ulSendOffset += sizeof(ulAreaNumber);

  memcpy(&pbData[ulSendOffset], &ulOffset, sizeof(ulOffset));
  ulSendOffset += sizeof(ulOffset);

  memcpy(&pbData[ulSendOffset], &ulTimeout, sizeof(ulTimeout));
  ulSendOffset += sizeof(ulTimeout);

  memcpy(&pbData[ulSendOffset], &ulDataLen, sizeof(ulDataLen));
  ulSendOffset += sizeof(ulDataLen);

  memcpy(&pbData[ulSendOffset], pvData, ulDataLen);
  ulSendOffset += ulDataLen;

  uint8_t*  pbRecvData    = NULL;
  uint32_t  ulRecvDataLen = 0;

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)ptHeader,
                                            ulRequestSize,
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            ulTimeout);

  if(CIFX_NO_ERROR == lRet)
  {
    PMARSHALLER_DATA_FRAME_HEADER_T ptMarshHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(pbRecvData);
    
    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(ptHeader, ptMarshHeader)) )
    {
      lRet = ptMarshHeader->ulError;
    }
        
    delete [] pbRecvData;
  }
  delete [] ptHeader;

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelIOReadSendData,
///   \param  hChannel        Handle of a channel
///   \param  ulAreaNumber    Number area 
///   \param  ulOffset        Data start offset 
///   \param  ulDataLen       Data length to write 
///   \param  pvData          Reference for the readed data 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelIOReadSendData( CIFXHANDLE    hChannel,
                                             uint32_t      ulAreaNumber,
                                             uint32_t      ulOffset,
                                             uint32_t      ulDataLen,
                                             void*         pvData)
{
  int32_t               lRet      = CIFX_INVALID_HANDLE;
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);



  uint8_t*                      pbRecvData      = NULL;
  uint32_t                      ulRecvDataLen   = 0;
  CHANNEL_IOREADSENDDATA_REQ_T  tReadSendREQ    = {0};

  tReadSendREQ.tData.ulArea       = ulAreaNumber;
  tReadSendREQ.tData.ulOffset     = ulOffset;
  tReadSendREQ.tData.ulDataLen    = ulDataLen;

  GenerateHeader( &tReadSendREQ.tHeader,
                  ptChannel->dwRemoteHandle,
                  MARSHALLER_CHANNEL_METHODID_IOREADSENDDATA,
                  true,
                  sizeof(tReadSendREQ.tData));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tReadSendREQ,
                                            sizeof(tReadSendREQ),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    PMARSHALLER_DATA_FRAME_HEADER_T ptMarshHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(pbRecvData);

    lRet = VerifyTransferResult(&tReadSendREQ.tHeader, ptMarshHeader);
    
    if ( (CIFX_NO_ERROR == lRet) ||
         (CIFX_DEV_NO_COM_FLAG == lRet) )
    {
      /* Check the user buffer size and copy at least the length of the user buffer */
      uint32_t ulCopySize = ptMarshHeader->ulDataSize;
      if(ulDataLen < ulCopySize)
      {
        ulCopySize = ulDataLen;
        lRet       = CIFX_BUFFER_TOO_SHORT;
      }
      memcpy( pvData, ptMarshHeader + 1, ulCopySize);
    }
    delete [] pbRecvData;
  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// ChannelReadWriteBlock,
///   \param  ptChannel       Channel information structure
///   \param  ulCmd           Command for block access
///   \param  ulOffset        Offset of the block access
///   \param  ulDataLen       Length of data (read / write) -> depends on ulCmd
///   \param  pvData          data pointer 
///   \param  ulMethodID      MethodID of the function call on device side 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::ChannelReadWriteBlock( PMARSH_CHANNEL_INFO_T ptChannel,
                                            uint32_t              ulCmd,
                                            uint32_t              ulOffset,
                                            uint32_t              ulDataLen,
                                            void*                 pvData,
                                            uint32_t              ulMethodID)
{
  int32_t                              lRet;

  switch(ulCmd)
  {
    case CIFX_CMD_READ_DATA:
      {
        CHANNEL_BLOCK_READ_REQ_T tBlockReadReq = {0};

        GenerateHeader( &tBlockReadReq.tHeader, 
                        ptChannel->dwRemoteHandle,
                        ulMethodID,
                        true,
                        sizeof(tBlockReadReq.tData));

        tBlockReadReq.tData.ulCmd     = ulCmd;
        tBlockReadReq.tData.ulOffset  = ulOffset;
        tBlockReadReq.tData.ulDatalen = ulDataLen;

        uint8_t*  pbRecvData    = NULL;
        uint32_t  ulRecvDataLen = 0;

        lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                                  (BYTE*)&tBlockReadReq,
                                                  sizeof(tBlockReadReq),
                                                  pbRecvData,
                                                  ulRecvDataLen,
                                                  m_pcEndpoint,
                                                  0,
                                                  CIFX_TO_CONT_PACKET);

        if(CIFX_NO_ERROR == lRet)
        {
          PCHANNEL_BLOCK_READ_CNF_T ptReadCnf = reinterpret_cast<PCHANNEL_BLOCK_READ_CNF_T>(pbRecvData);

          if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tBlockReadReq.tHeader, &ptReadCnf->tHeader)) )
          {
            memcpy(pvData, ptReadCnf->tData.abData, ulDataLen);
          }

          delete [] pbRecvData;
        }
      }
      break;

    case CIFX_CMD_WRITE_DATA:
      {
        PCHANNEL_BLOCK_WRITE_REQ_T ptBlockWriteReq = {0};

        uint32_t ulDataSendLen = sizeof(*ptBlockWriteReq) + ulDataLen;

        ptBlockWriteReq = (PCHANNEL_BLOCK_WRITE_REQ_T)new uint8_t[ulDataSendLen];

        GenerateHeader( &ptBlockWriteReq->tHeader, 
                        ptChannel->dwRemoteHandle, 
                        ulMethodID,
                        true, 
                        ulDataSendLen);

        ptBlockWriteReq->tData.ulCmd     = ulCmd;
        ptBlockWriteReq->tData.ulOffset  = ulOffset;
        ptBlockWriteReq->tData.ulDatalen = ulDataLen;
        memcpy(ptBlockWriteReq->tData.abData, pvData, ulDataLen);

        uint8_t*  pbRecvData    = NULL;
        uint32_t  ulRecvDataLen = 0;

        lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                                  (BYTE*)ptBlockWriteReq,
                                                  sizeof(*ptBlockWriteReq) +
                                                  ulDataLen,
                                                  pbRecvData,
                                                  ulRecvDataLen,
                                                  m_pcEndpoint,
                                                  0,
                                                  CIFX_TO_CONT_PACKET);

        if(CIFX_NO_ERROR == lRet)
        {
          PCHANNEL_BLOCK_READ_CNF_T ptReadCnf = reinterpret_cast<PCHANNEL_BLOCK_READ_CNF_T>(pbRecvData);

          if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&ptBlockWriteReq->tHeader, &ptReadCnf->tHeader)) )
          {
            lRet = ptReadCnf->tHeader.ulError;
          }

          delete [] pbRecvData;
        }
        delete [] ptBlockWriteReq;
      }
      break;

    default:
      lRet = CIFX_INVALID_COMMAND;
      break;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelIOReadSendData,
///   \param  hChannel        Handle of a channel
///   \param  ulCmd           Command
///   \param  ulOffset        Data start offset 
///   \param  ulDataLen       Data length to write 
///   \param  pvData          Reference for the readed data 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelControlBlock(CIFXHANDLE  hChannel,
                                          uint32_t    ulCmd,
                                          uint32_t    ulOffset,
                                          uint32_t    ulDataLen,
                                          void*       pvData)
{
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  return ChannelReadWriteBlock(ptChannel, ulCmd, ulOffset, ulDataLen, pvData, MARSHALLER_CHANNEL_METHODID_CONTROLBLOCK);
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelCommonStatusBlock,
///   \param  hChannel        Handle of a channel
///   \param  ulCmd           Command 
///   \param  ulOffset        Data start offset 
///   \param  ulDataLen       Data length to write 
///   \param  pvData          Reference for the readed data 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelCommonStatusBlock( CIFXHANDLE  hChannel,
                                                uint32_t    ulCmd,
                                                uint32_t    ulOffset,
                                                uint32_t    ulDataLen,
                                                void*       pvData)
{
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  return ChannelReadWriteBlock(ptChannel, ulCmd, ulOffset, ulDataLen, pvData, MARSHALLER_CHANNEL_METHODID_STATUSBLOCK);
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelExtendedStatusBlock,
///   \param  hChannel        Handle of a channel
///   \param  ulCmd           Command 
///   \param  ulOffset        Data start offset 
///   \param  ulDataLen       Data length to write 
///   \param  pvData          Reference for the readed data 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelExtendedStatusBlock( CIFXHANDLE  hChannel,
                                                  uint32_t    ulCmd,
                                                  uint32_t    ulOffset,
                                                  uint32_t    ulDataLen,
                                                  void*       pvData)
{
  PMARSH_CHANNEL_INFO_T ptChannel = reinterpret_cast<PMARSH_CHANNEL_INFO_T>(hChannel);

  return ChannelReadWriteBlock(ptChannel, ulCmd, ulOffset, ulDataLen, pvData, MARSHALLER_CHANNEL_METHODID_EXTSTATUSBLOCK);
}

/////////////////////////////////////////////////////////////////////////////
/// xChannelUserBlock,
///   \param  hChannel        Handle of a channel
///   \param  ulAreaNumber    Number of area
///   \param  ulCmd           Command 
///   \param  ulOffset        Data start offset 
///   \param  ulDataLen       Data length to write 
///   \param  pvData          Reference for the readed data 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::xChannelUserBlock( CIFXHANDLE  hChannel, uint32_t ulAreaNumber, uint32_t ulCmd, uint32_t ulOffset, uint32_t ulDataLen, void* pvData)
{
  UNREFERENCED_PARAMETER( hChannel);
  UNREFERENCED_PARAMETER( ulAreaNumber);
  UNREFERENCED_PARAMETER( ulCmd);
  UNREFERENCED_PARAMETER( ulOffset);
  UNREFERENCED_PARAMETER( ulDataLen);
  UNREFERENCED_PARAMETER( pvData);

  if(NULL == hChannel)
    return CIFX_INVALID_HANDLE;
  
  if(NULL == pvData)
    return CIFX_INVALID_POINTER;

	// UserBlock
  return CIFX_FUNCTION_NOT_AVAILABLE;
}

/////////////////////////////////////////////////////////////////////////////
/// Close the given channel
///   \param  ptChannel    Channel handle
///   \param  fSendRequest Object on target should be deleted
///   \param  fForceRemove Channel will be closed regardless of reference counter
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::CloseChannel(PMARSH_CHANNEL_INFO_T ptChannel, bool fSendRequest /* = true */, bool fForceRemove /* = false */)
{

  int32_t lRet  = CIFX_INVALID_HANDLE;

  EnterCriticalSection(&m_tDeviceMapLock);

  /* Search channel handle in our list */
  MARSH_DEVICE_MAP::iterator iterDevice = m_cmDevice.begin();

  bool fChannelFound = false;

  while( (iterDevice != m_cmDevice.end()) && !fChannelFound)
  {
    MARSH_CHANNEL_MAP::iterator iterChannel = iterDevice->second.begin();

    while(iterChannel != iterDevice->second.end())
    {
      if(iterChannel->second == ptChannel)
      {
        /* We've found our channel handle */

        /* Last one turns off the light ;) */
        if( fForceRemove ||
            (0 == InterlockedDecrement(&ptChannel->lRefCount)) )
        {
          /* Delete object on target */
          if(fSendRequest)
            DeleteDeviceObject(ptChannel->dwRemoteHandle, ptChannel->ulChannelNr);

          /* Remove all internal references */
          iterDevice->second.erase(iterChannel);

          delete ptChannel;
        }

        lRet = CIFX_NO_ERROR;
        fChannelFound = true;
        break;
      }

      iterChannel++;
    }

    iterDevice++;
  }

  LeaveCriticalSection(&m_tDeviceMapLock);
  
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Open a channel on remote end
///   \param  szDevice     Device/Board name
///   \param  dwChannelNr  Nr of the channel to open (CIFX_SYSTEM_DEVICE for sysdevice)
///   \param  phChannel    Returned handle
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::OpenChannel(std::string szDevice, DWORD dwChannelNr, CIFXHANDLE* phChannel)
{
  int32_t lRet      = CIFX_NO_ERROR;
  bool    fCreated  = false;

  PMARSH_CHANNEL_INFO_T ptChannel = NULL;

  // Check, channel was opened before 
  MARSH_DEVICE_MAP::iterator iterDevice = m_cmDevice.find(szDevice);
  
  if(iterDevice != m_cmDevice.end())
  {
    MARSH_CHANNEL_MAP::iterator iterChannel = iterDevice->second.find(dwChannelNr);

    if(iterChannel != iterDevice->second.end())
    {
      ptChannel = iterChannel->second;
    }
  }

  if(NULL == ptChannel )
  {
    // It's the first time that we open this channel 
    ptChannel                 = new MARSH_CHANNEL_INFO_T;
    ptChannel->dwRemoteHandle = 0;
    ptChannel->lRefCount      = 0;
    ptChannel->ulChannelNr    = dwChannelNr;
    fCreated                  = true;
  }

  if(0 == ptChannel->lRefCount)
  {
    uint32_t                   ulRequestSize   = 0;
    PMARSHALLER_DATA_FRAME_HEADER_T ptHeader        = NULL;
    const uint32_t                  ulDeviceNameLen = (uint32_t)szDevice.length();

    if(dwChannelNr == CIFX_SYSTEM_DEVICE)
    {
      ulRequestSize = (uint32_t) (sizeof(*ptHeader) +
                                  sizeof(DWORD)     +    /* ulBoardNameSize          */
                                  ulDeviceNameLen);      /* Boardname (NOT \0 terminated) */

      ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);

      size_t         ulOffset = 0;
      uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptHeader + 1);

      memcpy(&pbData[ulOffset], &ulDeviceNameLen, sizeof(ulDeviceNameLen));
      ulOffset += sizeof(ulDeviceNameLen);
      
      memcpy(&pbData[ulOffset], szDevice.c_str(), ulDeviceNameLen);
      ulOffset += ulDeviceNameLen;

    } else
    {
      ulRequestSize = (uint32_t)( sizeof(*ptHeader) +
                                  sizeof(DWORD)     +   /* ulBoardNameSize          */
                                  ulDeviceNameLen   +   /* Boardname (NOT \0 terminated) */
                                  sizeof(DWORD));       /* Channel number */

      ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);

      size_t         ulOffset = 0;
      uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptHeader + 1);

      memcpy(&pbData[ulOffset], &ulDeviceNameLen, sizeof(ulDeviceNameLen));
      ulOffset += sizeof(ulDeviceNameLen);

      memcpy(&pbData[ulOffset], szDevice.c_str(), ulDeviceNameLen);
      ulOffset += ulDeviceNameLen;

      memcpy(&pbData[ulOffset], &dwChannelNr, sizeof(dwChannelNr));
      ulOffset += sizeof(dwChannelNr);
    }

    uint8_t* pbRecvData    = NULL;
    uint32_t  ulRecvDataLen = 0;

    GenerateHeader(ptHeader, 
                    m_dwDriverHandle, 
                    (dwChannelNr == CIFX_SYSTEM_DEVICE) ? MARSHALLER_DRV_METHODID_OPENSYSDEV :
                                                          MARSHALLER_DRV_METHODID_OPENCHANNEL,
                    true, 
                    ulRequestSize - sizeof(*ptHeader));

    lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                              (BYTE*)ptHeader,
                                              ulRequestSize,
                                              pbRecvData,
                                              ulRecvDataLen,
                                              m_pcEndpoint,
                                              0,
                                              CIFX_TO_CONT_PACKET);

    if(CIFX_NO_ERROR == lRet)
    {
      PCF_CREATEINSTANCE_CNF_T ptCnf = reinterpret_cast<PCF_CREATEINSTANCE_CNF_T>(pbRecvData);

      if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(ptHeader, &ptCnf->tHeader)) )
      {
        ptChannel->dwRemoteHandle = ptCnf->tData.uHandle.ulHandle;

        // We got a new channel handle for the marshaller connection, so save this in openchannel 
        //    iterChannel = AddChannel( szDevice, ptChannel);
      }
      delete [] pbRecvData;
    }
    delete [] ptHeader;
  }

  if(CIFX_NO_ERROR == lRet)
  {
    // The device exist, so use the entry
    if(fCreated)
    {
      if(iterDevice == m_cmDevice.end())
      {
        MARSH_CHANNEL_MAP cEmptyChannelList;
        m_cmDevice[szDevice] = cEmptyChannelList;

        iterDevice = m_cmDevice.find(szDevice);
      }

      iterDevice->second[dwChannelNr] = ptChannel;
    }

    InterlockedIncrement(&ptChannel->lRefCount);
    *phChannel = ptChannel;

  } else if(fCreated)
  {
    delete ptChannel;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// PutPacket, put a packet into send queue, to send this packet 
///   \param  ptChannel    Reference of channel manage structure
///   \param  ptPacket     Reference of the sendable packet 
///   \param  ulTimeout    Timeout for this function call
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::PutPacket(PMARSH_CHANNEL_INFO_T ptChannel, CIFX_PACKET* ptPacket, uint32_t ulTimeout)
{
  int32_t                         lRet;
  uint32_t                        ulRequestSize   = 0;
  PMARSHALLER_DATA_FRAME_HEADER_T ptHeader        = NULL;
  uint32_t                        ulSendDataSize  = ptPacket->tHeader.ulLen + sizeof(ptPacket->tHeader);
  

  ulRequestSize = sizeof(*ptHeader) +
                  sizeof(DWORD)     +    /* ulSendSize  */
                  ulSendDataSize    +    /* abData[]    */
                  sizeof(DWORD);         /* ulTimeout   */

  ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);
  memset(ptHeader, 0, ulRequestSize);

  size_t         ulOffset = 0;
  uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptHeader + 1);

  memcpy(&pbData[ulOffset], &ulSendDataSize, sizeof(ulSendDataSize));
  ulOffset += sizeof(ulSendDataSize);

  memcpy(&pbData[ulOffset], ptPacket, ulSendDataSize);
  ulOffset += ulSendDataSize;

  memcpy(&pbData[ulOffset], &ulTimeout, sizeof(ulTimeout));
  ulOffset += sizeof(ulTimeout);

  uint8_t* pbRecvData    = NULL;
  uint32_t  ulRecvDataLen = 0;

  GenerateHeader(ptHeader, 
                 ptChannel->dwRemoteHandle,
                 (ptChannel->ulChannelNr == CIFX_SYSTEM_DEVICE) ? MARSHALLER_SYSDEV_METHODID_PUTPACKET :
                                                                  MARSHALLER_CHANNEL_METHODID_PUTPACKET,
                 true,
                 ulRequestSize - sizeof(*ptHeader));

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)ptHeader,
                                            ulRequestSize,
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            ulTimeout);

  if(CIFX_NO_ERROR == lRet)
  {
    PCF_CREATEINSTANCE_CNF_T ptCnf = reinterpret_cast<PCF_CREATEINSTANCE_CNF_T>(pbRecvData);

    lRet = VerifyTransferResult(ptHeader, &ptCnf->tHeader);

    delete [] pbRecvData;
  }
  delete [] ptHeader;

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// GetPacket, Get a received packet from queue 
///   \param ptChannel    Reference of channel manage structure
///   \param ptPacket     Reference for the receviced packet 
///   \param ulPacketLen  Length of the receive buffer
///   \param ulTimeout    Timeout for this function call
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::GetPacket(PMARSH_CHANNEL_INFO_T ptChannel, CIFX_PACKET* ptPacket, uint32_t ulPacketLen, uint32_t ulTimeout)
{
  int32_t                 lRet       = CIFX_NO_ERROR;
  CHANNEL_GETPACKET_REQ_T tGetPacket = {0};

  uint8_t* pbRecvData    = NULL;
  uint32_t ulRecvDataLen = 0;

  GenerateHeader( &tGetPacket.tHeader, 
                  ptChannel->dwRemoteHandle,
                 (ptChannel->ulChannelNr == CIFX_SYSTEM_DEVICE) ? MARSHALLER_SYSDEV_METHODID_GETPACKET : 
                                                                  MARSHALLER_CHANNEL_METHODID_GETPACKET,
                 true,
                 sizeof(tGetPacket.tData));

  tGetPacket.tData.ulSize    = ulPacketLen;
  tGetPacket.tData.ulTimeout = ulTimeout;                      
  
  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tGetPacket,
                                            sizeof(tGetPacket),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            ulTimeout);

  if(CIFX_NO_ERROR == lRet)
  {
    MARSHALLER_DATA_FRAME_HEADER_T* ptCnf = reinterpret_cast<MARSHALLER_DATA_FRAME_HEADER_T*>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tGetPacket.tHeader, ptCnf)) )
    {
      CIFX_PACKET* ptRecvPacket = reinterpret_cast<CIFX_PACKET*>(ptCnf + 1);

      uint32_t ulCopySize = ptRecvPacket->tHeader.ulLen + sizeof(CIFX_PACKET_HEADER);
      if(ulPacketLen < ulCopySize)
      {
        ulCopySize = ulPacketLen;
        lRet = CIFX_BUFFER_TOO_SHORT;
      }

      memcpy(ptPacket, ptRecvPacket, ulCopySize);
    }
    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// GetMBXState, Get State of mail box 
///   \param ptChannel    Reference of channel manage structure
///   \param pulRecvCnt   Reference for the receive counter value 
///   \param pulSendCnt   Reference for the value, for sendable packets (space in send queue)
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::GetMbxState(PMARSH_CHANNEL_INFO_T ptChannel, uint32_t* pulRecvCnt, uint32_t* pulSendCnt)
{
  int32_t  lRet;
  uint8_t* pbRecvData    = NULL;
  uint32_t ulRecvDataLen = 0;
  MARSHALLER_DATA_FRAME_HEADER_T tGetMbxState = {0};

  GenerateHeader(&tGetMbxState, 
                 ptChannel->dwRemoteHandle,
                 (ptChannel->ulChannelNr == CIFX_SYSTEM_DEVICE) ? MARSHALLER_SYSDEV_METHODID_GETMBXSTATE : 
                                                                  MARSHALLER_CHANNEL_METHODID_GETMBXSTATE,
                 true,
                 0);

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tGetMbxState,
                                            sizeof(tGetMbxState),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
     PCHANNEL_GETMBXSTATE_CNF_T ptCnf = reinterpret_cast<PCHANNEL_GETMBXSTATE_CNF_T>(pbRecvData);

    if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(&tGetMbxState, &ptCnf->tHeader)) )
    {
      *pulRecvCnt = ptCnf->tData.ulRecvPktCnt;
      *pulSendCnt = ptCnf->tData.ulSendPktCnt;
    }

    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// OpenDriver, Send a OpenDriver request to signal 
///             driver object should be open 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::OpenDriver( void)
{
  int32_t                        lRet        = CIFX_NO_ERROR;
  MARSHALLER_DATA_FRAME_HEADER_T tDriverOpen = {0};

  uint8_t*  pbRecvData    = NULL;
  uint32_t  ulRecvDataLen = 0;

  GenerateHeader(&tDriverOpen, 
                 m_dwDriverHandle, 
                 MARSHALLER_DRV_METHODID_OPEN,
                 true, 
                 0);

  lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                            (BYTE*)&tDriverOpen,
                                            sizeof(tDriverOpen),
                                            pbRecvData,
                                            ulRecvDataLen,
                                            m_pcEndpoint,
                                            0,
                                            CIFX_TO_CONT_PACKET);

  if(CIFX_NO_ERROR == lRet)
  {
    MARSHALLER_DATA_FRAME_HEADER_T* ptCnf = reinterpret_cast<MARSHALLER_DATA_FRAME_HEADER_T*>(pbRecvData);

    lRet = VerifyTransferResult(&tDriverOpen, ptCnf);

    delete [] pbRecvData;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Get information from the device (channel) 
///   \param ptDevice       Reference of a marshaller channel structur 
///   \param ulCmd          Command of the ordered information structure
///   \param ulSize         Size of the buffer for the recevice data 
///   \param pvInfo         Reference for the ordered data (information)
///   \param ulType         Type of channel (sysdevice / channel) 
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::MethodeInfo( PMARSH_CHANNEL_INFO_T ptDevice,
                                  uint32_t              ulCmd,
                                  uint32_t              ulSize,
                                  void*                 pvInfo,
                                  uint32_t              ulType)
{
  int32_t                         lRet            = CIFX_NO_ERROR;
  bool                            fCall           = false;
  uint32_t                        ulRequestSize   = 0;
  uint32_t                        ulMarshallerCmd = 0;
  PMARSHALLER_DATA_FRAME_HEADER_T ptHeader        = NULL;
  
  
  switch(ulType)
  {
    case CHANNEL_TYPE_XSYSDEVICE:
    {
      // HANDLE xSysdeviceInfo call
      ulRequestSize = sizeof(*ptHeader) + 
                      sizeof(DWORD)     +    /* ulCmd       */
                      sizeof(DWORD);         /* ulSize, of the returned structure  */


      ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);
      // Clean the allocated memory 
      memset(ptHeader, 0, ulRequestSize);

      size_t         ulOffset = 0;
      uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptHeader + 1);

      memcpy(&pbData[ulOffset], &ulCmd, sizeof(ulCmd));
      ulOffset += sizeof(ulCmd);

      memcpy(&pbData[ulOffset], &ulSize, sizeof(ulSize));
      ulOffset += sizeof(ulSize);

      ulMarshallerCmd = MARSHALLER_SYSDEV_METHODID_INFO;
      fCall = true;
    }
    break;

    case CHANNEL_TYPE_CHANNEL:
    {
      // HANDLE xChannelInfo call, ignore the ulCmd param
      ulRequestSize = sizeof(*ptHeader) + 
                      sizeof(DWORD);         /* ulSize, of the returned structure  */

      ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);
      // Clean the allocated memory 
      memset(ptHeader, 0, ulRequestSize);

      size_t         ulOffset = 0;
      uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptHeader + 1);

      memcpy(&pbData[ulOffset], &ulSize, sizeof(ulSize));
      ulOffset += sizeof(ulSize);
      ulMarshallerCmd = MARSHALLER_CHANNEL_METHODID_INFO;

      fCall = true;
    }
    break;

    default:
      lRet = CIFX_INVALID_PARAMETER;
      break;
  }

  if(fCall)
  {

    uint8_t* pbRecvData    = NULL;
    uint32_t  ulRecvDataLen = 0;

    GenerateHeader( ptHeader,
                    ptDevice->dwRemoteHandle,
                    ulMarshallerCmd,
                    true,
                    ulRequestSize - sizeof(*ptHeader));

    lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                              (BYTE*)ptHeader,
                                              ulRequestSize,
                                              pbRecvData,
                                              ulRecvDataLen,
                                              m_pcEndpoint,
                                              0,
                                              CIFX_TO_CONT_PACKET);

    if(CIFX_NO_ERROR == lRet)
    {

      PCF_CREATEINSTANCE_CNF_T ptCnf = reinterpret_cast<PCF_CREATEINSTANCE_CNF_T>(pbRecvData);

      if ( CIFX_NO_ERROR == (lRet = VerifyTransferResult(ptHeader, &ptCnf->tHeader)) )
      {
        // Copy return info structure
        memcpy( pvInfo, &ptCnf->tData , ulSize);
      }

      delete [] pbRecvData;
    }

    delete [] ptHeader;

  }
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Reset the device, independend on Mode and device (channel)
///   \param ptDevice       Reference of a marshaller channel structure 
///   \param ulResetMode    Reset mode
///   \param ulType         Size of the buffer for the recevice data 
///   \param ulTimeout      Timeout for the function call
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CMarshaller::Reset( PMARSH_CHANNEL_INFO_T ptDevice,
                            uint32_t              ulResetMode,
                            uint32_t              ulType,
                            uint32_t              ulTimeout)
{
  EnterCriticalSection(&m_tcsResetLock);

  int32_t                         lRet            = CIFX_NO_ERROR;
  bool                            fCall           = false;
  uint32_t                        ulRequestSize   = 0;
  uint32_t                        ulMarshallerCmd = 0;
  PMARSHALLER_DATA_FRAME_HEADER_T ptHeader        = NULL;

  switch(ulType)
  {
    // xSysdeviceReset 
    case CHANNEL_TYPE_XSYSDEVICE:
    {
      ulRequestSize = sizeof(*ptHeader) + 
                      sizeof(DWORD);        /* ulTimeout in ms to wait for reset to complete */

      ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);
      // Clean the allocated memory 
      memset(ptHeader, 0, ulRequestSize);

      size_t         ulOffset = 0;
      uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptHeader + 1);

      memcpy(&pbData[ulOffset], &ulTimeout, sizeof(ulTimeout));
      ulOffset += sizeof(ulTimeout);
      ulMarshallerCmd = MARSHALLER_SYSDEV_METHODID_RESET;

      fCall = true;
    }
    break;

    // xChannelReset
    case CHANNEL_TYPE_CHANNEL:
    {
      ulRequestSize = sizeof(*ptHeader) + 
                      sizeof(DWORD)     +   /* ulResetMode, special mode of reset */
                      sizeof(DWORD);        /* ulTimeout in ms for function call  */

      ptHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(new uint8_t[ulRequestSize]);
      // Clean the allocated memory 
      memset(ptHeader, 0, ulRequestSize);
      
      size_t         ulOffset = 0;
      uint8_t* pbData   = reinterpret_cast<uint8_t*>(ptHeader + 1);

      memcpy(&pbData[ulOffset], &ulResetMode, sizeof(ulResetMode));
      ulOffset += sizeof(ulResetMode);

      memcpy(&pbData[ulOffset], &ulTimeout, sizeof(ulTimeout));
      ulOffset += sizeof(ulTimeout);
      ulMarshallerCmd = MARSHALLER_CHANNEL_METHODID_RESET;
      fCall = true;
    }
    break;

    default:
      lRet = CIFX_INVALID_PARAMETER;
      break;
  }

  if(fCall)
  {
    // Send reset packet
    uint8_t*  pbRecvData    = NULL;
    uint32_t  ulRecvDataLen = 0;

    GenerateHeader(ptHeader,
                  ptDevice->dwRemoteHandle,
                  ulMarshallerCmd,
                  true,
                  ulRequestSize - sizeof(*ptHeader));
    

    lRet = m_pcTransportLayer->TransferPacket(HIL_TRANSPORT_TYPE_MARSHALLER,
                                              (BYTE*)ptHeader,
                                              ulRequestSize,
                                              pbRecvData,
                                              ulRecvDataLen,
                                              m_pcEndpoint,
                                              0,
                                              ulTimeout);

    if(CIFX_NO_ERROR == lRet)
    {
      PMARSHALLER_DATA_FRAME_HEADER_T ptMarshHeader = reinterpret_cast<PMARSHALLER_DATA_FRAME_HEADER_T>(pbRecvData);
      
      lRet = VerifyTransferResult(ptHeader, ptMarshHeader);

      delete [] pbRecvData;
    }

    delete [] ptHeader;
  }

  LeaveCriticalSection(&m_tcsResetLock);

  return lRet;
}
